/*
 * Copyright (C) 2002 Scott Smith (trckjunky@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "compat.h"

#include "dvdauthor.h"
#include "da-internal.h"

static const char RCSID[]="$Id: //depot/dvdauthor/src/dvdvob.c#25 $";


struct colorremap {
    int newcolors[16];
    int state,curoffs,maxlen,nextoffs,skip;
};

struct vscani {
    int lastrefsect;
    int firstgop,firsttemporal,lastadjust,adjustfields;
};

static int timeline[19]={1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,
                         20,60,120,240};

#define BIGWRITEBUFLEN (16*2048)
static unsigned char bigwritebuf[BIGWRITEBUFLEN];
static int writebufpos=0;
static int writefile=-1;

static unsigned char videoslidebuf[15]={255,255,255,255, 255,255,255, 0,0,0,0, 0,0,0,0};

static int calcpts(struct vobgroup *va,int basepts,int nfields)
{
    return basepts+(nfields*getframepts(va))/2;
}

static int findnextvideo(struct vob *va, int cur, int dir)
{
    // find next (dir=1) or previous(dir=-1) vobu with video
    int i, numvobus;
    
    numvobus = va->numvobus;
    switch(dir){
    case 1:  // forward
        for(i = cur+1; i < numvobus; i++) if(va->vi[i].hasvideo) return i;
        return -1;
    case -1: // backward
        for(i = cur-1; i > -1; i--) if(va->vi[i].hasvideo) return i;
        return -1;
    default:
        // ??
        return -1;
    }
}

static int findaudsect(struct vob *va,int aind,int pts0,int pts1)
{
    struct audchannel *ach=&va->audch[aind];
    int l=0,h=ach->numaudpts-1;

    if( h<l )
        return -1;
    while(h>l) {
        int m=(l+h+1)/2;
        if( pts0<ach->audpts[m].pts[0] )
            h=m-1;
        else
            l=m;
    }
    if( ach->audpts[l].pts[0] > pts1 )
        return -1;
    return ach->audpts[l].sect;
}

static int findvobubysect(struct vob *va,int sect)
{
    int l=0,h=va->numvobus-1;

    if( h<0 )
        return -1;
    if( sect<va->vi[0].sector )
        return -1;
    while(l<h) {
        int m=(l+h+1)/2;
        if( sect < va->vi[m].sector )
            h=m-1;
        else
            l=m;
    }
    return l;
}

static int findspuidx(struct vob *va,int ach,int pts0)
{
    int l=0,h=va->audch[ach].numaudpts-1;

    if( h<l )
        return -1;
    while(h>l) {
        int m=(l+h+1)/2;
        if( pts0<va->audch[ach].audpts[m].pts[0] )
            h=m-1;
        else
            l=m;
    }
    return l;
}

static unsigned int getsect(struct vob *va,int curvobnum,int jumpvobnum,int skip,unsigned notfound)
{
    if( skip )
        skip=0x40000000;
    if( jumpvobnum < 0 || jumpvobnum >= va->numvobus || 
        va->vi[jumpvobnum].vobcellid != va->vi[curvobnum].vobcellid )
        return notfound|skip;
    return abs(va->vi[jumpvobnum].sector-va->vi[curvobnum].sector)
        |0x80000000|skip;
}

static int readscr(const unsigned char *buf)
{
    return ((buf[0]&0x18)<<27)|
            ((buf[0]&3)<<28)|
            (buf[1]<<20)|
            ((buf[2]&0xf8)<<12)|
            ((buf[2]&3)<<13)|
            (buf[3]<<5)|
            ((buf[4]&0xf8)>>3);
}

static void writescr(unsigned char *buf,unsigned int scr)
{
    buf[0]=((scr>>27)&0x18)|((scr>>28)&3)|68;
    buf[1]=scr>>20;
    buf[2]=((scr>>12)&0xf8)|((scr>>13)&3)|4;
    buf[3]=scr>>5;
    buf[4]=((scr<<3)&0xf8)|(buf[4]&7);
}

static int readpts(const unsigned char *buf)
{
    int a1,a2,a3,pts;
    a1=(buf[0]&0xe)>>1;
    a2=((buf[1]<<8)|buf[2])>>1;
    a3=((buf[3]<<8)|buf[4])>>1;
    pts=(((int64_t)a1)<<30)|
        (a2<<15)|
        a3;
    return pts;
}


static void writepts(unsigned char *buf,unsigned int pts)
{
    buf[0]=((pts>>29)&0xe)|(buf[0]&0xf1); // this preserves the PTS / DTS / PTSwDTS top bits
    write2(buf+1,(pts>>14)|1);
    write2(buf+3,(pts<<1)|1);
}

static int findbutton(struct pgc *pg,char *dest,int dflt)
{
    int i;

    if( !dest )
        return dflt;
    for( i=0; i<pg->numbuttons; i++ )
        if( !strcmp(pg->buttons[i].name,dest) )
            return i+1;
    return dflt;
}

static void transpose_ts(unsigned char *buf,int tsoffs)
{
    // pack scr
    if( buf[0] == 0 &&
        buf[1] == 0 &&
        buf[2] == 1 &&
        buf[3] == 0xba )
    {
        writescr(buf+4,readscr(buf+4)+tsoffs);

        // video/audio?
        // pts?
        if( buf[14] == 0 &&
            buf[15] == 0 &&
            buf[16] == 1 &&
            (buf[17]==0xbd || (buf[17]>=0xc0 && buf[17]<=0xef)) &&
            (buf[21] & 128))
        {
            writepts(buf+23,readpts(buf+23)+tsoffs);
            // dts?
            if( buf[21] & 64 ) {
                writepts(buf+28,readpts(buf+28)+tsoffs);
            }
        }
    }
}

static int mpa_valid(unsigned char *b)
{
    unsigned int v=(b[0]<<24)|(b[1]<<16)|(b[2]<<8)|b[3];
    int t;

    // sync, mpeg1, layer2, 48khz
    if( (v&0xFFFE0C00) != 0xFFFC0400 )
        return 0;
    // bitrate 1..14
    t=(v>>12)&15;
    if( t==0 || t==15 )
        return 0;
    // emphasis reserved
    if( (v&3)==2 )
        return 0;
    return 1;
}


static int mpa_len(unsigned char *b)
{
    static int bitratetable[16]={0,32,48,56,64,80,96,112,128,160,192,224,256,320,384,0};
    int padding=(b[2]>>1)&1;
    int bitrate=bitratetable[(b[2]>>4)&15];
    
    return 3*bitrate+padding; // 144*bitrate/sampling; 144/48=3
}

static void writeflush()
{
    if( !writebufpos ) return;
    if( write(writefile,bigwritebuf,writebufpos) != writebufpos ) {
        fprintf(stderr,"ERR:  Error writing data\n");
        exit(1);
    }
    writebufpos=0;
}

static unsigned char *writegrabbuf()
{
    unsigned char *buf;
    if( writebufpos == BIGWRITEBUFLEN )
        writeflush();
    buf=bigwritebuf+writebufpos;
    writebufpos+=2048;
    return buf;
}

static void writeundo()
{
    writebufpos-=2048;
}

static void writeclose()
{
    writeflush();
    if( writefile!=-1 ) {
        close(writefile);
        writefile=-1;
    }
}

static void writeopen(const char *newname)
{
    writefile=open(newname,O_CREAT|O_WRONLY|O_BINARY,0666);
    if( writefile < 0 ) {
        fprintf(stderr,"ERR:  Error opening %s: %s\n",newname,strerror(errno));
        exit(1);
    }
}

static void closelastref(struct vobuinfo *thisvi,struct vscani *vsi,int cursect)
{
    if( vsi->lastrefsect && thisvi->numref < 3 ) {
        thisvi->lastrefsect[thisvi->numref++]=cursect;
        vsi->lastrefsect=0;
    }
}

// this function is allowed to update buf[7] and gaurantee it will end up
// in the output stream
// prevbytesect is the sector for the byte immediately preceding buf[0]
static void scanvideoptr(struct vobgroup *va,unsigned char *buf,struct vobuinfo *thisvi,int prevbytesect,struct vscani *vsi)
{
    if( buf[0]==0 &&
        buf[1]==0 &&
        buf[2]==1 ) {
        switch(buf[3]) {
        case 0: {
            int ptype=(buf[5]>>3)&7;
            int temporal=(buf[4]<<2)|(buf[5]>>6);

            closelastref(thisvi,vsi,prevbytesect);
            if( vsi->firsttemporal==-1 )
                vsi->firsttemporal=temporal;
            vsi->lastadjust=( temporal < vsi->firsttemporal );
            if( ptype == 1 || ptype == 2 ) // I or P frame
                vsi->lastrefsect=1;

            if( va->vd.vmpeg==VM_MPEG1) {
                thisvi->numfields+=2;
                if(vsi->lastadjust && vsi->firstgop==2)
                    thisvi->firstIfield+=2;
            }

            // fprintf(stderr,"INFO: frame type %d, tempref=%d, prevsect=%d\n",ptype,temporal,prevbytesect);
            break;
        } 

        case 0xb3: { // sequence header
            int hsize,vsize,aspect,frame,newaspect;
            char sizestring[30];
            
            closelastref(thisvi,vsi,prevbytesect);

            hsize=(buf[4]<<4)|(buf[5]>>4);
            vsize=((buf[5]<<8)&0xf00)|buf[6];
            aspect=buf[7]>>4;
            frame=buf[7]&0xf;

            vobgroup_set_video_framerate(va,frame);
            switch(frame) {
            case 1:
            case 4:
            case 7:
                vobgroup_set_video_attr(va,VIDEO_FORMAT,"ntsc");
                break;

            case 3:
            case 6:
                vobgroup_set_video_attr(va,VIDEO_FORMAT,"pal");
                break;

            default:
                fprintf(stderr,"WARN: unknown frame rate %d\n",frame);
                break;
            }
           
            sprintf(sizestring,"%dx%d",hsize,vsize);
            vobgroup_set_video_attr(va,VIDEO_RESOLUTION,sizestring);

            if( va->vd.vmpeg==VM_MPEG1) {
                switch( aspect ) {
                case 3:
                    vobgroup_set_video_attr(va,VIDEO_ASPECT,"16:9");
                    vobgroup_set_video_attr(va,VIDEO_FORMAT,"pal");
                    break;
                case 6:
                    vobgroup_set_video_attr(va,VIDEO_ASPECT,"16:9");
                    vobgroup_set_video_attr(va,VIDEO_FORMAT,"ntsc");
                    break;
                case 8:
                    vobgroup_set_video_attr(va,VIDEO_ASPECT,"4:3");
                    vobgroup_set_video_attr(va,VIDEO_FORMAT,"pal");
                    break;
                case 12:
                    vobgroup_set_video_attr(va,VIDEO_ASPECT,"4:3");
                    vobgroup_set_video_attr(va,VIDEO_FORMAT,"ntsc");
                    break;
                default:
                    fprintf(stderr,"WARN: unknown mpeg1 aspect ratio %d\n",aspect);
                    break;
                }
                newaspect=3+
                    (va->vd.vaspect==VA_4x3)*5+
                    (va->vd.vformat==VF_NTSC)*3;
                if( newaspect==11 ) newaspect++;
                buf[7]=(buf[7]&0xf)|(newaspect<<4); // reset the aspect ratio
            } else if( va->vd.vmpeg==VM_MPEG2 ) {
                if( aspect==2 )
                    vobgroup_set_video_attr(va,VIDEO_ASPECT,"4:3");
                else if( aspect==3 )
                    vobgroup_set_video_attr(va,VIDEO_ASPECT,"16:9");
                else
                    fprintf(stderr,"WARN: unknown mpeg2 aspect ratio %d\n",aspect);
                buf[7]=(buf[7]&0xf)|(va->vd.vaspect==VA_4x3?2:3)<<4; // reset the aspect ratio
            }
            break;
        }

        case 0xb5: { // extension header
            vobgroup_set_video_attr(va,VIDEO_MPEG,"mpeg2");
            switch(buf[4]&0xF0) {
            case 0x10: // sequence extension
                closelastref(thisvi,vsi,prevbytesect);
                break;

            case 0x20: // sequence display extension
                closelastref(thisvi,vsi,prevbytesect);
                switch(buf[4]&0xE) {
                case 2: vobgroup_set_video_attr(va,VIDEO_FORMAT,"pal"); break;
                case 4: vobgroup_set_video_attr(va,VIDEO_FORMAT,"ntsc"); break;
                    // case 6: // secam
                    // case 10: // unspecified
                }
                break;

            case 0x80: { // picture coding extension
                int padj=1; // default field pic
                
                if( (buf[6]&3)==3 ) padj++; // adj for frame pic
                if( buf[7]&2 )      padj++; // adj for repeat flag
                
                thisvi->numfields+=padj;
                if(vsi->lastadjust && vsi->firstgop==2)
                    thisvi->firstIfield+=padj;
                // fprintf(stderr,"INFO: repeat flag=%d, cursect=%d\n",buf[7]&2,cursect);
                break;
            }

            }
            break;
        }
            
        case 0xb7: { // sequence end code
            thisvi->hasseqend=1;
            break;
        }

        case 0xb8: // gop header
            closelastref(thisvi,vsi,prevbytesect);
            if( vsi->firstgop==1 ) {
                vsi->firstgop=2;
                vsi->firsttemporal=-1;
                vsi->lastadjust=0;
                vsi->adjustfields=0;
            } else if( vsi->firstgop==2 ) {
                vsi->firstgop=0;
            }
            break;

        /*
        case 0xb8: { // gop header
            int hr,mi,se,fr;

            hr=(buf[4]>>2)&31;
            mi=((buf[4]&3)<<4)|(buf[5]>>4);
            se=((buf[5]&7)<<3)|(buf[6]>>5);
            fr=((buf[6]&31)<<1)|(buf[7]>>7);
            fprintf(stderr,"INFO: GOP header, %d:%02d:%02d:%02d, drop=%d\n",hr,mi,se,fr,(buf[4]>>7));
            break;
        }
        */

        }
    }
}

static void scanvideoframe(struct vobgroup *va,unsigned char *buf,struct vobuinfo *thisvi,int cursect,int prevsect,struct vscani *vsi)
{
    int i,f=0x17+buf[0x16],l=0x14+buf[0x12]*256+buf[0x13];
    int mpf;
    struct vobuinfo oldtvi;
    struct vscani oldvsi;

    if( l-f<8 ) {
        memcpy(videoslidebuf+7,buf+f,l-f);
        for( i=0; i<l-f; i++ )
            scanvideoptr(va,videoslidebuf+i,thisvi,prevsect,vsi);
        memcpy(buf+f,videoslidebuf+7,l-f);
        memset(videoslidebuf,255,7);
        return;
    }

 rescan:
    mpf=va->vd.vmpeg;
    oldtvi=*thisvi;
    oldvsi=*vsi;

    // copy the first 7 bytes to use with the prev 7 bytes in hdr detection
    memcpy(videoslidebuf+7,buf+f,8); // we scan the first header using the slide buffer
    for( i=0; i<=7; i++ )
        scanvideoptr(va,videoslidebuf+i,thisvi,prevsect,vsi);
    memcpy(buf+f,videoslidebuf+7,8);

    // quickly scan all but the last 7 bytes for a hdr
    // buf[f]... was already scanned in the videoslidebuffer to give the correct sector
    for( i=f+1; i<l-7; i++ ) {
        if( buf[i]==0 && buf[i+1]==0 && buf[i+2]==1 )
            scanvideoptr(va,buf+i,thisvi,cursect,vsi);
    }
    if( !va->vd.vmpeg )
        vobgroup_set_video_attr(va,VIDEO_MPEG,"mpeg1");
    // if the mpeg version changed, then rerun scanvideoframe, because
    // scanvideoptr updates the aspect ratio in the sequence header
    if( mpf != va->vd.vmpeg ) {
        *thisvi=oldtvi; // we must undo all the frame pointer changes
        *vsi=oldvsi;
        goto rescan;
    }

    // use the last 7 bytes in the next iteration
    memcpy(videoslidebuf,buf+l-7,7);
}

static void finishvideoscan(struct vobgroup *va,int vob,int prevsect,struct vscani *vsi)
{
    struct vobuinfo *lastvi=&va->vobs[vob]->vi[va->vobs[vob]->numvobus-1];
    int i;

    memset(videoslidebuf+7,0,7);
    for( i=0; i<7; i++ )
        scanvideoptr(va,videoslidebuf+i,lastvi,prevsect,vsi);
    memset(videoslidebuf,255,7);
    closelastref(lastvi,vsi,prevsect);
}

static void printpts(int pts)
{
    fprintf(stderr,"%d.%03d",pts/90000,(pts/90)%1000);
}

static FILE *openvob(char *f,int *ispipe)
{
    FILE *h;
    int l=strlen(f);

    if( l>0 && f[l-1]=='|' ) {
        char *str;
        int i;

        f[l-1]=0;
        str=(char *)malloc(l*2+1+10);
        strcpy(str,"sh -c \"");
        l=strlen(str);
        for( i=0; f[i]; i++ ) {
            if( f[i]=='\"' || f[i]=='\'' )
                str[l++]='\\';
            str[l++]=f[i];
        }
        str[l]=0;
        strcpy(str+l,"\"");
        h=popen(str,"r");
        free(str);
        ispipe[0]=1;
    } else if( !strcmp(f,"-") ) {
        h=stdin;
        ispipe[0]=2;
    } else {
        h=fopen(f,"rb");
        ispipe[0]=0;
    }
    if( !h ) {
        fprintf(stderr,"ERR:  Error opening %s: %s\n",f,strerror(errno));
        exit(1);
    }
    return h;
}

enum { CR_BEGIN0,   CR_BEGIN1,    CR_BEGIN2,    CR_BEGIN3, CR_SKIP0,
       CR_SKIP1,    CR_NEXTOFFS0, CR_NEXTOFFS1, CR_WAIT,   CR_CMD,
       CR_SKIPWAIT, CR_COL0,      CR_COL1};

static char *readpstr(char *b,int *i)
{
    char *s=strdup(b+i[0]);
    i[0]+=strlen(s)+1;
    return s;
}

static void initremap(struct colorremap *cr)
{
    int i;

    for( i=0; i<16; i++ )
        cr->newcolors[i]=i;
    cr->state=CR_BEGIN0;
}

static void remapbyte(struct colorremap *cr,unsigned char *b)
{
    b[0]=cr->newcolors[b[0]&15]|(cr->newcolors[b[0]>>4]<<4);
}

static void procremap(struct colorremap *cr,unsigned char *b,int l,int *timespan)
{
    while(l) {
        // fprintf(stderr,"INFO: state=%d, byte=%02x (%d)\n",cr->state,*b,*b);
        switch(cr->state) {
        case CR_BEGIN0: cr->curoffs=0; cr->skip=0; cr->maxlen=b[0]*256; cr->state=CR_BEGIN1; break;
        case CR_BEGIN1: cr->maxlen+=b[0]; cr->state=CR_BEGIN2; break;
        case CR_BEGIN2: cr->nextoffs=b[0]*256; cr->state=CR_BEGIN3; break;
        case CR_BEGIN3: cr->nextoffs+=b[0]; cr->state=CR_WAIT; break;

        case CR_WAIT:
            if( cr->curoffs==cr->maxlen ) {
                cr->state=CR_BEGIN0;
                continue; // execute BEGIN0 right away
            }
            if( cr->curoffs!=cr->nextoffs )
                break;
            cr->state=CR_SKIP0;
            // fall through to CR_SKIP0
        case CR_SKIP0: *timespan+=1024*b[0]*256; cr->state=CR_SKIP1; break;
        case CR_SKIP1: *timespan+=1024*b[0]; cr->state=CR_NEXTOFFS0; break;
        case CR_NEXTOFFS0: cr->nextoffs=b[0]*256; cr->state=CR_NEXTOFFS1; break;
        case CR_NEXTOFFS1: cr->nextoffs+=b[0]; cr->state=CR_CMD; break;
        case CR_SKIPWAIT:
            if( cr->skip ) {
                cr->skip--;
                break;
            }
            cr->state=CR_CMD;
            // fall through to CR_CMD
        case CR_CMD:
            switch(*b) {
            case 0: break;
            case 1: break;
            case 2: break;
            case 3: cr->state=CR_COL0; break;
            case 4: cr->skip=2; cr->state=CR_SKIPWAIT; break;
            case 5: cr->skip=6; cr->state=CR_SKIPWAIT; break;
            case 6: cr->skip=4; cr->state=CR_SKIPWAIT; break;
            case 255: cr->state=CR_WAIT; break;
            default:
                fprintf(stderr,"ERR: procremap encountered unknown subtitle command: %d\n",*b);
                exit(1);
            }
            break;

        case CR_COL0:
            remapbyte(cr,b);
            cr->state=CR_COL1;
            break;

        case CR_COL1:
            remapbyte(cr,b);
            cr->state=CR_CMD;
            break;
           
        default:
            assert(0);
        }
        cr->curoffs++;
        b++;
        l--;
    }
}

static void printvobustatus(struct vobgroup *va,int cursect)
{
    int i,j,total=0,nv=0;

    for( j=0; j<va->numvobs; j++ ) {
        int tm=-1;
        for( i=va->vobs[j]->numvobus-1; i>=0; i-- )
            if( va->vobs[j]->vi[i].pts[1]!=-1 ) {
                tm=va->vobs[j]->vi[i].pts[1];
                break;
            }
        if( tm!=-1 )
            tm-=va->vobs[j]->vi[0].pts[0];
        else
            tm=0;
        total+=tm;
        nv+=va->vobs[j]->numvobus;
    }

    fprintf(stderr,"STAT: VOBU %d at %dMB, %d PGCS, %d:%02d:%02d\r",nv,cursect/512,va->numallpgcs,total/324000000,(total%324000000)/5400000,(total%5400000)/90000);
}

static void audio_scan_ac3(struct audchannel *ach,unsigned char *buf,int sof,int len)
{
    u_int32_t parse;
    int acmod,lfeon,nch=0;
    char attr[4];

    if( sof+8>len ) // check if there's room to parse all the interesting info
        return;
    if( buf[sof]!=0x0b || buf[sof+1]!=0x77 ) // verify ac3 sync
        return;
    parse=read4(buf+sof+4);
    parse<<=8;
    // check bsid
    if( (parse>>27)!=8 && (parse>>27)!=6 ) // must be v6 or v8
        return;
    parse<<=5;
    // ignore bsmod
    parse<<=3;
    // acmod gives # of channels
    acmod=(parse>>29);
    parse<<=3;
    // skip cmixlev?
    if( (acmod&1) && (acmod!=1) )
        parse<<=2;
    // skip surmixlev?
    if( acmod&4 )
        parse<<=2;
    // dsurmod
    if( acmod==2 ) {
        if( (parse>>30)==2 ) 
            audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_DOLBY,"surround");
        // else if( (parse>>30)==1 )
        // audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_DOLBY,"nosurround");
        parse<<=2;
    }
    // lfeon
    lfeon=(parse>>31);

    // calc # channels
    switch(acmod) {
    case 0: nch=2; break;
    case 1: nch=1; break;
    case 2: nch=2; break;
    case 3: nch=3; break;
    case 4: nch=3; break;
    case 5: nch=4; break;
    case 6: nch=4; break;
    case 7: nch=5; break;
    }
    if( lfeon ) nch++;
    sprintf(attr,"%dch",nch);
    audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_CHANNELS,attr);
}

static void audio_scan_dts(struct audchannel *ach,unsigned char *buf,int sof,int len)
{
}

static void audio_scan_pcm(struct audchannel *ach,unsigned char *buf,int len)
{
    char attr[4];

    switch(buf[1]>>6) {
    case 0: audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_QUANT,"16bps"); break;
    case 1: audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_QUANT,"20bps"); break;
    case 2: audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_QUANT,"24bps"); break;
    }
    sprintf(attr,"%dch",(buf[1]&7)+1);
    audiodesc_set_audio_attr(&ach->ad,&ach->adwarn,AUDIO_CHANNELS,attr);
}

int FindVobus(char *fbase,struct vobgroup *va,int ismenu)
{
    unsigned char *buf;
    FILE *h;
    int cursect=0,fsect=-1,vnum,outnum=-ismenu+1;
    int ispipe,vobid=0;
    int mp2hdr[8];
    struct colorremap *crs;
    
    crs=malloc(sizeof(struct colorremap)*32);
    for( vnum=0; vnum<va->numvobs; vnum++ ) {
        int i;
        int hadfirstvobu=0,backoffs=0;
        struct vob *s=va->vobs[vnum];
        // struct pgc *p=va->pgcs[pnum];
        int prevvidsect=-1;
        struct vscani vsi;

        vsi.lastrefsect=0;
        for( i=0; i<32; i++ )
            initremap(crs+i);

        vobid++;
        s->vobid=vobid;
        vsi.firstgop=1;

        fprintf(stderr,"\nSTAT: Processing %s...\n",s->fname);
        h=openvob(s->fname,&ispipe);
        memset(mp2hdr,0,8*sizeof(int));
        while(1) {
            if( fsect == 524272 ) {
                writeclose();
                if( outnum<=0 ) {
                    fprintf(stderr,"\nERR:  Menu VOB reached 1gb\n");
                    exit(1);
                }
                outnum++;
                fsect=-1;
            }
            buf=writegrabbuf();
            if( 2048 != fread(buf,1,2048,h) ) {
                writeundo();
                break;
            }
            if( buf[14]==0 && buf[15]==0 && buf[16]==1 && buf[17]==0xbe && !strcmp(buf+20,"dvdauthor-data")) {
                // private dvdauthor data, should be removed from the final stream
                int i=35;
                if( buf[i]!=1 ) {
                    fprintf(stderr,"ERR: dvd info packet is version %d\n",buf[i]);
                    exit(1);
                }
                switch(buf[i+1]) { // packet type

                case 1: // subtitle/menu color and button information
                {
                    int st=buf[i+2]&31;
                    i+=3;
                    i+=8; // skip start pts and end pts
                    while(buf[i]!=0xff) {
                        switch(buf[i]) {
                        case 1: // new colormap
                        {
                            int j;
                            for( j=0; j<buf[i+1]; j++ ) {
                                int c=(buf[i+2+3*j]<<16)|(buf[i+3+3*j]<<8)|(buf[i+4+3*j]), k;
                                for( k=0; k<16; k++ )
                                    if( s->p->ci->colors[k]==c ) {
                                        crs[st].newcolors[j]=k;
                                        break;
                                    }
                                if( k<16 ) continue;
                                for( k=0; k<16; k++ )
                                    if( s->p->ci->colors[k]==0x1000000 ) {
                                        s->p->ci->colors[k]=c;
                                        crs[st].newcolors[j]=k;
                                        break;
                                    }
                                if( k<16 ) continue;
                                fprintf(stderr,"ERR: color map full, unable to allocate new colors.\n");
                                exit(1);
                            }
                            for( ; j<16; j++ )
                                crs[st].newcolors[j]=j;
                            i+=2+3*buf[i+1];
                            break;
                        }
                        case 2: // new buttoncoli
                        {
                            int j;

                            memcpy(s->buttoncoli,buf+i+2,buf[i+1]*8);
                            for( j=0; j<buf[i+1]; j++ ) {
                                remapbyte(&crs[st],s->buttoncoli+j*8+0);
                                remapbyte(&crs[st],s->buttoncoli+j*8+1);
                                remapbyte(&crs[st],s->buttoncoli+j*8+4);
                                remapbyte(&crs[st],s->buttoncoli+j*8+5);
                            }
                            i+=2+8*buf[i+1];
                            break;
                        }
                        case 3: // button position information
                        {
                            int j,k;

                            k=buf[i+1];
                            i+=2;
                            for( j=0; j<k; j++ ) {
                                struct button *b;
                                char *st=readpstr(buf,&i);
                                    
                                if( !findbutton(s->p,st,0) ) {
                                    fprintf(stderr,"ERR:  Cannot find button '%s' as referenced by the subtitle\n",st);
                                    exit(1);
                                }
                                b=&s->p->buttons[findbutton(s->p,st,0)-1];

                                i+=2; // skip modifier
                                b->autoaction=buf[i++];
                                if( !b->autoaction ) {
                                    b->grp=buf[i];
                                    b->x1=read2(buf+i+1);
                                    b->y1=read2(buf+i+3);
                                    b->x2=read2(buf+i+5);
                                    b->y2=read2(buf+i+7);
                                    i+=9;
                                    // up down left right
                                    b->up=readpstr(buf,&i);
                                    b->down=readpstr(buf,&i);
                                    b->left=readpstr(buf,&i);
                                    b->right=readpstr(buf,&i);
                                }
                            }
                            break;
                        }
                        default:
                            fprintf(stderr,"ERR: dvd info packet command within subtitle: %d\n",buf[i]);
                            exit(1);
                        }
                    }

                    break;
                }
                        
                default:
                    fprintf(stderr,"ERR: unknown dvd info packet type: %d\n",buf[i+1]);
                    exit(1);
                }

                writeundo();
                continue;
            }
            if( !hadfirstvobu && buf[0]==0 && buf[1]==0 && buf[2]==1 && buf[3]==0xba )
                backoffs=readscr(buf+4);
            transpose_ts(buf,-backoffs);
            if( fsect == -1 ) {
                char newname[200];
                fsect=0;
                if( outnum>=0 )
                    sprintf(newname,"%s_%d.VOB",fbase,outnum);
                else
                    strcpy(newname,fbase);
                writeopen(newname);
            }
            if( buf[14] == 0 &&
                buf[15] == 0 &&
                buf[16] == 1 &&
                buf[17] == 0xbb ) // system header
            {
                if( buf[38] == 0 &&
                    buf[39] == 0 &&
                    buf[40] == 1 &&
                    buf[41] == 0xbf && // 1st private2
                    buf[1024] == 0 &&
                    buf[1025] == 0 &&
                    buf[1026] == 1 &&
                    buf[1027] == 0xbf ) // 2nd private2
                {
                    struct vobuinfo *vi;
                    if( s->numvobus )
                        finishvideoscan(va,vnum,prevvidsect,&vsi);
                    // fprintf(stderr,"INFO: vobu\n");
                    hadfirstvobu=1;
                    s->numvobus++;
                    if( s->numvobus > s->maxvobus ) {
                        if( !s->maxvobus )
                            s->maxvobus=1;
                        else
                            s->maxvobus<<=1;
                        s->vi=(struct vobuinfo *)realloc(s->vi,s->maxvobus*sizeof(struct vobuinfo));
                    }
                    vi=&s->vi[s->numvobus-1];
                    memset(vi,0,sizeof(struct vobuinfo));
                    vi->sector=cursect;
                    vi->fsect=fsect;
                    vi->fnum=outnum;
                    if( s->numvobus==1 ) {
                        vi->pts[0]=-1;
                        vi->pts[1]=-1;
                        vi->firstfield=0;
                        vi->firstIfield=0;
                    } else {
                        vi->firstfield=s->vi[s->numvobus-2].firstfield+s->vi[s->numvobus-2].numfields;
                        vi->firstIfield=vi->firstfield;
                        vi->pts[0]=calcpts(va,s->vi[0].pts[0],vi->firstfield);
                        s->vi[s->numvobus-2].pts[1]=vi->pts[0]; // backfill pts
                        vi->pts[1]=vi->pts[0];
                    }
                    vi->numfields=0;
                    vi->numref=0;
                    vi->hasseqend=0;
                    vi->hasvideo=0;
                    vi->sectdata=(unsigned char *)malloc(2048);
                    memcpy(s->vi[s->numvobus-1].sectdata,buf,2048);
                    if( !(s->numvobus&15) )
                        printvobustatus(va,cursect);
                    vsi.lastrefsect=0;
                    vsi.firstgop=1;
                } else {
                    fprintf(stderr,"WARN: System header found, but PCI/DSI information is not where expected\n\t(make sure your system header is 18 bytes!)\n");
                }
            }
            if( !hadfirstvobu ) {
                fprintf(stderr,"WARN: Skipping sector, waiting for first VOBU...\n");
                writeundo();
                continue;
            }
            s->vi[s->numvobus-1].lastsector=cursect;
            if( buf[0] == 0 &&
                buf[1] == 0 &&
                buf[2] == 1 &&
                buf[3] == 0xba &&
                buf[14] == 0 &&
                buf[15] == 0 &&
                buf[16] == 1 &&
                buf[17] == 0xe0 ) { // video
                struct vobuinfo *vi=&s->vi[s->numvobus-1];
                vi->hasvideo=1;
                scanvideoframe(va,buf,vi,cursect,prevvidsect,&vsi);
                if( (buf[21] & 128) && s->numvobus==1 && vi->pts[0]==-1 ) { // check whether there's a pts
                    vi->pts[0]=readpts(buf+23);
                }
                prevvidsect=cursect;
            }
            if( buf[0] == 0 &&
                buf[1] == 0 &&
                buf[2] == 1 &&
                buf[3] == 0xba &&
                buf[14] == 0 &&
                buf[15] == 0 &&
                buf[16] == 1 &&
                ((buf[17] & 0xf8) == 0xc0 || buf[17]==0xbd) &&
                buf[21] & 128 ) {
                int pts0=readpts(buf+23),pts1;
                int dptr=buf[22]+23,endop=read2(buf+18)+20;
                int audch;
                pts1=pts0;
                if( buf[17]==0xbd ) {
                    int sid=buf[dptr],offs=read2(buf+dptr+2);

                    switch(sid&0xf8) {
                    case 0x20:                          // subpicture
                    case 0x28:                          // subpicture
                    case 0x30:                          // subpicture
                    case 0x38: audch=sid; break;        // subpicture
                    case 0x80:                          // ac3
                        pts1+=2880*buf[dptr+1];
                        audch=sid&7;
                        audio_scan_ac3(&s->audch[audch],buf+dptr+4,offs-1,endop-(dptr+4));
                        break;
                    case 0x88:                          // dts
                        audch=24|(sid&7);
                        audio_scan_dts(&s->audch[audch],buf+dptr+4,offs-1,endop-(dptr+4));
                        break;
                    case 0xa0:                          // pcm
                        pts1+=150*buf[dptr+1];
                        audch=16|(sid&7);
                        audio_scan_pcm(&s->audch[audch],buf+dptr+4,endop-(dptr+4));
                        break;
                    default:   audch=-1; break;         // unknown
                    }
                } else {
                    int len=endop-dptr;
                    int index=buf[17]&7;
                    audch=8|index;                      // mp2
                    if( mp2hdr[index]<0 ) mp2hdr[index]=0;
                    while(mp2hdr[index]+4<=len) {
                        if( !mpa_valid(buf+dptr+mp2hdr[index]) ) {
                            mp2hdr[index]++;
                            continue;
                        }
                        mp2hdr[index]+=mpa_len(buf+dptr+mp2hdr[index]);
                        pts1+=2160;
                    }
                    mp2hdr[index]-=len;
                }
                // fprintf(stderr,"aud ch=%d pts %d - %d\n",audch,pts0,pts1);
                // fprintf(stderr,"pts[%d] %d (%02x %02x %02x %02x %02x)\n",va->numaudpts,pts,buf[23],buf[24],buf[25],buf[26],buf[27]);
                if( audch<0 || audch>=64 ) {
                    fprintf(stderr,"WARN: Invalid audio channel %d\n",audch);
                } else {
                    struct audchannel *ach=&s->audch[audch];

                    if( ach->numaudpts>=ach->maxaudpts ) {
                        if( ach->maxaudpts )
                            ach->maxaudpts<<=1;
                        else
                            ach->maxaudpts=1;
                        ach->audpts=(struct audpts *)realloc(ach->audpts,ach->maxaudpts*sizeof(struct audpts));
                    }
                    ach->audpts[ach->numaudpts].pts[0]=pts0;
                    ach->audpts[ach->numaudpts].pts[1]=pts1;
                    ach->audpts[ach->numaudpts].sect=cursect;
                    ach->numaudpts++;
                }
            }
            // the following code scans subtitle code in order to
            // remap the colors and update the end pts
            if( buf[0] == 0 &&
                buf[1] == 0 &&
                buf[2] == 1 &&
                buf[3] == 0xba &&
                buf[14] == 0 &&
                buf[15] == 0 &&
                buf[16] == 1 &&
                buf[17] == 0xbd) {
                int dptr=buf[22]+23,ml=read2(buf+18)+20;
                int st=buf[dptr];
                dptr++;
                if( (st&0xf8)==0x20 ) { // subtitle
                    procremap(&crs[st&31],buf+dptr,ml-dptr,&s->audch[st].audpts[s->audch[st].numaudpts-1].pts[1]);
                }
            }
            cursect++;
            fsect++;
        }
        switch(ispipe) {
        case 0: fclose(h); break;
        case 1: pclose(h); break;
        case 2: break;
        }
        if( s->numvobus ) {
            int i;
                
            finishvideoscan(va,vnum,prevvidsect,&vsi);
            // find any non-video vobus
            for( i=1; i<s->numvobus; i++ ) {
                if( s->vi[i].pts[0]==-1 ) {
                    s->vi[i].pts[0]=s->vi[i-1].pts[0];
                    s->vi[i].pts[1]=s->vi[i-1].pts[0];
                }
            }
            s->vi[s->numvobus-1].pts[1]=calcpts(va,s->vi[0].pts[0],s->vi[s->numvobus-1].firstfield+s->vi[s->numvobus-1].numfields);
            fprintf(stderr,"\nINFO: Video pts = ");
            printpts(s->vi[0].pts[0]);
            fprintf(stderr," .. ");
            printpts(s->vi[s->numvobus-1].pts[1]);
            for( i=0; i<64; i++ ) {
                struct audchannel *ach=&s->audch[i];

                if( ach->numaudpts ) {
                    fprintf(stderr,"\nINFO: Audio[%d] pts = ",i);
                    printpts(ach->audpts[0].pts[0]);
                    fprintf(stderr," .. ");
                    printpts(ach->audpts[ach->numaudpts-1].pts[1]);
                }
            }
            fprintf(stderr,"\n");
        }
    }
    writeclose();
    printvobustatus(va,cursect);
    fprintf(stderr,"\n");
    free(crs);
    return 1;
}

static int getvobuclosedpts(struct vobgroup *pg,struct vob *va,int i)
{
    return calcpts(pg,va->vi[i].pts[0],va->vi[i].firstIfield-va->vi[i].firstfield);
}

static int findvobu_closed(struct vobgroup *pg,struct vob *va,int pts)
{
    int l=0,h=va->numvobus-1;

    if( h<0 )
        return -1;
    pts+=getvobuclosedpts(pg,va,0)+900; // we add 900 so that the timestamp can be up to 1/100th of a second off
    if( pts<getvobuclosedpts(pg,va,0) )
        return -1;
    if( pts>va->vi[h].pts[1] )
        return h+1;
    while(l<h) {
        int m=(l+h+1)/2;
        if( pts < getvobuclosedpts(pg,va,m) )
            h=m-1;
        else
            l=m;
    }
    if( l+1<va->numvobus &&
        abs(pts-getvobuclosedpts(pg,va,l+1))<abs(pts-getvobuclosedpts(pg,va,l)) )
        l++;
    return l;
}

#if 0
static void printtime(int t)
{
    if( t<0 )
        fprintf(stderr,"N/A");
    else
        fprintf(stderr,"%d:%02d:%02d.%03d",
                (t/90/1000/60/60),
                (t/90/1000/60)%60,
                (t/90/1000)%60,
                (t/90)%1000);
}

static int scanclosed(struct vob *v,int i,int d)
{
    while(i>=0 && i<v->numvobus) {
        if( v->vi[i].firstIfield==v->vi[i].firstfield )
            return v->vi[i].pts[0]-v->vi[0].pts[0];
        i+=d;
    }
    return -1;
}
#endif

void MarkChapters(struct vobgroup *va)
{
    int i,j,k,lastcellid;

    // mark start and stop points
    lastcellid=-1;
    for( i=0; i<va->numallpgcs; i++ )
        for( j=0; j<va->allpgcs[i]->numsources; j++ ) {
            struct source *s=va->allpgcs[i]->sources[j];

            for( k=0; k<s->numcells; k++ ) {
                int v;
                
                v=findvobu_closed(va,s->vob,s->cells[k].startpts);
                if( v>=0 && v<s->vob->numvobus )
                    s->vob->vi[v].vobcellid=1;
                s->cells[k].scellid=v;

                if( lastcellid!=v &&
                    s->vob->vi[v].firstIfield!=s->vob->vi[v].firstfield) {
                    fprintf(stderr,"WARN: GOP is not closed on cell %d of source %s of pgc %d\n",k+1,s->fname,i+1);
                }

                if( s->cells[k].endpts>=0 ) {
                    v=findvobu_closed(va,s->vob,s->cells[k].endpts);
                    if( v>=0 && v<s->vob->numvobus )
                        s->vob->vi[v].vobcellid=1;
                } else
                    v=s->vob->numvobus;
                s->cells[k].ecellid=v;

                lastcellid=v;
            }
        }
    // tally up actual cells
    for( i=0; i<va->numvobs; i++ ) {
        int cellvobu=0;
        int cellid=va->vobs[i]->vobid*256;
        va->vobs[i]->vi[0].vobcellid=1;
        for( j=0; j<va->vobs[i]->numvobus; j++ ) {
            struct vobuinfo *v=&va->vobs[i]->vi[j];
            if( v->vobcellid ) {
                cellid++;
                cellvobu=j;
#if 0
                if( v->firstfield!=v->firstIfield ) {
                    int adjpts=calcpts(va,0,v->firstIfield-v->firstfield);
                    v->pts[0]+=adjpts;
                    fprintf(stderr,"WARN: GOP is not closed on cell boundary: requested ");
                    printtime(v->pts[0]-va->vobs[i]->vi[0].pts[0]);
                    fprintf(stderr,"; next forward: ");
                    printtime(scanclosed(va->vobs[i],j,1));
                    fprintf(stderr,"; next backward: ");
                    printtime(scanclosed(va->vobs[i],j,-1));
                    fprintf(stderr,"\n");
                    if(j) va->vobs[i]->vi[j-1].pts[1]+=adjpts;
                }
#endif
            }
            v->vobcellid=cellid;
            v->firstvobuincell=cellvobu;
        }
        cellvobu=va->vobs[i]->numvobus-1;
        for( j=cellvobu; j>=0; j-- ) {
            struct vobuinfo *v=&va->vobs[i]->vi[j];
            v->lastvobuincell=cellvobu;
            if(v->firstvobuincell==j )
                cellvobu=j-1;
        }
        cellid&=255;
        va->vobs[i]->numcells=cellid;
    }

    // now compute scellid and ecellid
    for( i=0; i<va->numallpgcs; i++ )
        for( j=0; j<va->allpgcs[i]->numsources; j++ ) {
            struct source *s=va->allpgcs[i]->sources[j];

            for( k=0; k<s->numcells; k++ ) {
                struct cell *c=&s->cells[k];

                if( c->scellid<0 )
                    c->scellid=1;
                else if( c->scellid<s->vob->numvobus )
                    c->scellid=s->vob->vi[c->scellid].vobcellid&255;
                else
                    c->scellid=s->vob->numcells+1;

                if( c->ecellid<0 )
                    c->ecellid=1;
                else if( c->ecellid<s->vob->numvobus )
                    c->ecellid=s->vob->vi[c->ecellid].vobcellid&255;
                else
                    c->ecellid=s->vob->numcells+1;

                va->allpgcs[i]->numcells+=c->ecellid-c->scellid;
                if( c->scellid!=c->ecellid && c->ischapter ) {
                    va->allpgcs[i]->numprograms++;
                    if( c->ischapter==1 )
                        va->allpgcs[i]->numchapters++;
                }
            }
        }
}

static int getcellaudiopts(struct vobgroup *va,int vcid,int ach)
{
    assert( (vcid&255)==1 );
    return va->vobs[(vcid>>8)-1]->audch[ach].audpts[0].pts[0];
}

static int calcaudiogap(struct vobgroup *va,int vcid,int ach)
{
    int gap=0,foundgap=0;
    int i,j,k,l;

    for( i=0; i<va->numallpgcs; i++ ) {
        int lastvcid=-1;
        struct pgc *p=va->allpgcs[i];

        for( j=0; j<p->numsources; j++ ) {
            struct source *s=p->sources[j];
            for( k=0; k<s->numcells; k++ ) {
                struct cell *c=&s->cells[k];

                for( l=c->scellid; l<c->ecellid; l++ ) {
                    int cid=l+s->vob->vobid*256;
                    if( cid==vcid ) {
                        int vi1,gp,v2,a2;

                        if( lastvcid==-1 )
                            gp=0;
                        else if( l==1 && (lastvcid&255)==va->vobs[(lastvcid>>8)-1]->numcells ) {
                            struct vob *lv=va->vobs[(lastvcid>>8)-1];
                            vi1=findcellvobu(s->vob,l);
                            gp=lv->vi[lv->numvobus-1].pts[1]-
                                lv->audch[ach].audpts[lv->audch[ach].numaudpts-1].pts[1];
                            v2=s->vob->vi[vi1].pts[0];
                            a2=getcellaudiopts(va,cid,ach);
                            gp-=v2-a2;
                        } else if( cid==lastvcid+1 ) {
                            gp=0;
                        } else {
                            fprintf(stderr,"WARN: Do not know how to compute the audio gap in this situation\n");
                            gp=0;
                        }
                        if( foundgap && gp!=gap ) {
                            fprintf(stderr,"WARN: Multiple paths entering cell requiring gaps %d and %d, picking lower\n",gp,gap);
                            if( gp<gap ) gap=gp;
                        } else {
                            foundgap=1;
                            gap=gp;
                        }
                    }
                    lastvcid=cid;
                }
            }
        }
    }
    if( foundgap )
        return gap;
    else
        return 0;
}

void FixVobus(char *fbase,struct vobgroup *va,int ismenu)
{
    int h=-1;
    int i,j,pn,scr,fnum=-2;
    int vff,vrew,totvob,curvob;
    unsigned char *buf;

    totvob=0;
    for( pn=0; pn<va->numvobs; pn++ )
        totvob+=va->vobs[pn]->numvobus;
    curvob=0;

    for( pn=0; pn<va->numvobs; pn++ ) {
        struct vob *p=va->vobs[pn];

        for( i=0; i<p->numvobus; i++ ) {
            struct vobuinfo *vi=&p->vi[i];

            if( vi->fnum!=fnum ) {
                char fname[200];
                if( h >= 0 )
                    close(h);
                fnum=vi->fnum;
                if( fnum==-1 )
                    strcpy(fname,fbase);
                else
                    sprintf(fname,"%s_%d.VOB",fbase,fnum);
                h=open(fname,O_WRONLY|O_BINARY);
                if( h < 0 ) {
                    fprintf(stderr,"\nERR:  Error opening %s: %s\n",fname,strerror(errno));
                    exit(1);
                }
            }
            buf=vi->sectdata;
            memset(buf+0x2d,0,0x400-0x2d);
            memset(buf+0x407,0,0x7ff-0x407);

            scr=readscr(buf+4);

            buf[0x2c]=0;
            write4(buf+0x2d,vi->sector);
            write4(buf+0x39,vi->pts[0]); // vobu_s_ptm
            write4(buf+0x3d,vi->pts[1]); // vobu_e_ptm
            if( vi->hasseqend ) // if sequence_end_code
                write4(buf+0x41,vi->pts[1]); // vobu_se_e_ptm
            write4(buf+0x45,buildtimeeven(va,vi->pts[0]-p->vi[vi->firstvobuincell].pts[0])); // total guess
                
            if( p->p->numbuttons ) {
                struct pgc *pg=p->p;

                write2(buf+0x8d,1);
                write4(buf+0x8f,p->vi[0].pts[0]);
                write4(buf+0x93,-1);
                write4(buf+0x97,-1);
                write2(buf+0x9b,0x1000);
                buf[0x9e]=pg->numbuttons;
                buf[0x9f]=pg->numbuttons;
                memcpy(buf+0xa3,p->buttoncoli,24);
                for( j=0; j<pg->numbuttons; j++ ) {
                    struct button *b=pg->buttons+j;
                    if( b->autoaction ) {
                        buf[0xbb+j*18+3]=64;
                    } else {
                        buf[0xbb+j*18+0]=(b->grp*64)|(b->x1>>4);
                        buf[0xbb+j*18+1]=(b->x1<<4)|(b->x2>>8);
                        buf[0xbb+j*18+2]=b->x2;
                        buf[0xbb+j*18+3]=(b->y1>>4);
                        buf[0xbb+j*18+4]=(b->y1<<4)|(b->y2>>8);
                        buf[0xbb+j*18+5]=b->y2;
                        buf[0xbb+j*18+6]=findbutton(pg,b->up,(j==0)?pg->numbuttons:j);
                        buf[0xbb+j*18+7]=findbutton(pg,b->down,(j+1==pg->numbuttons)?1:j+2);
                        buf[0xbb+j*18+8]=findbutton(pg,b->left,(j==0)?pg->numbuttons:j);
                        buf[0xbb+j*18+9]=findbutton(pg,b->right,(j+1==pg->numbuttons)?1:j+2);
                    }
                    write8(buf+0xbb+j*18+10,0x71,0x01,0x00,0x0F,0x00,j+1,0x00,0x0d); // g[0]=j && linktailpgc
                }
            }

            buf[0x406]=1;
            write4(buf+0x407,scr);
            write4(buf+0x40b,vi->sector); // current lbn
            for( j=0; j<vi->numref; j++ )
                write4(buf+0x413+j*4,vi->lastrefsect[j]-vi->sector);
            write2(buf+0x41f,vi->vobcellid>>8);
            buf[0x422]=vi->vobcellid;
            write4(buf+0x423,read4(buf+0x45));
            write4(buf+0x433,p->vi[0].pts[0]);
            write4(buf+0x437,p->vi[p->numvobus-1].pts[1]);
            // audio gap
            if( vi->firstvobuincell==i ) {
                for( j=0; j<va->numaudiotracks; j++ ) {
                    int ach=getaudch(va,j), gap;

                    gap=calcaudiogap(va,vi->vobcellid,ach);
                    // fprintf(stderr,"AUDIO GAP[%d]=%d\n",j,gap);
                    if( gap > 0 ) {
                        int a2=getcellaudiopts(va,vi->vobcellid,ach);
                        if( a2<gap ) {
                            fprintf(stderr,"WARN: Forced to truncate audio channel %d gap from %d to %d\n",j+1,gap,a2);
                            gap=a2;
                        }
                        write4(buf+0x43b+j*16,a2-gap);
                        write4(buf+0x443+j*16,gap);
                    } else if( gap < 0 ) {
                        fprintf(stderr,"WARN: Audio channel %d gap is negative (%d) on VOB %s cell %d\n",j+1,gap,p->fname,vi->vobcellid&255);
                    }
                }
            }

            write4(buf+0x4f1,getsect(p,i,findnextvideo(p,i,1),0,0xbfffffff));
            write4(buf+0x541,getsect(p,i,i+1,0,0x3fffffff));
            write4(buf+0x545,getsect(p,i,i-1,0,0x3fffffff));
            write4(buf+0x595,getsect(p,i,findnextvideo(p,i,-1),0,0xbfffffff));
            for( j=0; j<va->numaudiotracks; j++ ) {
                int s=findaudsect(p,getaudch(va,j),vi->pts[0],vi->pts[1]);
                if( s>=0 ) {
                    s=s-vi->sector;
                    if( s > 0x1fff || s < -(0x1fff)) {
                        fprintf(stderr,"\nWARN: audio sector out of range: %d (vobu #%d, pts %d)\n",s,i,vi->pts[0]);
                        s=0;
                    }
                    if( s < 0 )
                        s=(-s)|0x8000;
                } else
                    s=0x3fff;
                write2(buf+0x599+j*2,s);
            }
            for( j=0; j<va->numsubpicturetracks; j++ ) {
                struct audchannel *ach=&p->audch[j|32];
                int s;

                if( ach->numaudpts ) {
                    int id=findspuidx(p,j|32,vi->pts[0]);
                    
                    // if overlaps A, point to A
                    // else if (A before here or doesn't exist) and (B after here or doesn't exist), point to here
                    // else point to B

                    if( id>=0 && 
                        ach->audpts[id].pts[0]<vi->pts[1] &&
                        ach->audpts[id].pts[1]>=vi->pts[0] )
                        s=findvobubysect(p,ach->audpts[id].sect);
                    else if( (id<0 || ach->audpts[id].pts[1]<vi->pts[0]) &&
                             (id+1==ach->numaudpts || ach->audpts[id+1].pts[0]>=vi->pts[1]) )
                        s=i;
                    else
                        s=findvobubysect(p,ach->audpts[id+1].sect);
                    id=(s<i);
                    s=getsect(p,i,s,0,0x7fffffff)&0x7fffffff;
                    if(!s)
                        s=0x7fffffff;
                    if(s!=0x7fffffff && id)
                        s|=0x80000000;
                } else
                    s=0;
                write4(buf+0x5a9+j*4,s);
            }
            write4(buf+0x40f,vi->lastsector-vi->sector);
            vff=i;
            vrew=i;
            for( j=0; j<19; j++ ) {
                int nff,nrew;

                nff=findvobu(p,vi->pts[0]+timeline[j]*45000,
                             vi->firstvobuincell,
                             vi->lastvobuincell);
                // a hack -- the last vobu in the cell shouldn't have any forward ptrs
                if( i==vi->lastvobuincell )
                    nff=i+1;
                nrew=findvobu(p,vi->pts[0]-timeline[j]*45000,
                              vi->firstvobuincell,
                              vi->lastvobuincell);
                write4(buf+0x53d-j*4,getsect(p,i,nff,j>=15 && nff>vff+1,0x3fffffff));
                write4(buf+0x549+j*4,getsect(p,i,nrew,j>=15 && nrew<vrew-1,0x3fffffff));
                vff=nff;
                vrew=nrew;
            }

            lseek(h,vi->fsect*2048,SEEK_SET);
            write(h,buf,2048);
            curvob++;
            if( !(curvob&15) ) 
                fprintf(stderr,"STAT: fixing VOBU at %dMB (%d/%d, %d%%)\r",vi->sector/512,curvob+1,totvob,curvob*100/totvob);
        }
    }
    if( h>=0 )
        close(h);
    if( totvob>0 )
        fprintf(stderr,"STAT: fixed %d VOBUS                         ",totvob);
    fprintf(stderr,"\n");
}

