/*
 * Copyright (C) 2002 Scott Smith (trckjunky@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>
#include <dvdread/nav_read.h>

#include "compat.h"

static const char RCSID[]="$Id: //depot/dvdauthor/src/dvdunauthor.c#6 $";

#define BIGBLOCKSECT 512
#define BIGBLOCKLEN (DVD_VIDEO_LB_LEN*BIGBLOCKSECT)
static unsigned char bigblock[BIGBLOCKLEN];

static int numtitlesets=0;
static FILE *xml;

struct cellstarttime {
    int vob,cell,pts;
} *cellstarttimes=0;

int numcst=0;

static void addcst(int v,int c,int p)
{
    int i;

    for( i=0; i<numcst; i++ )
        if( cellstarttimes[i].vob==v && cellstarttimes[i].cell==c )
            return;
    cellstarttimes=realloc(cellstarttimes,(numcst+1)*sizeof(struct cellstarttime));
    cellstarttimes[numcst].vob=v;
    cellstarttimes[numcst].cell=c;
    cellstarttimes[numcst].pts=p;
    numcst++;
}

static int vobexists(cell_adr_t *cells,int numcells,int vobid)
{
    int i;

    for( i=0; i<numcells; i++ )
        if( cells[i].vob_id==vobid )
            return 1;
    return 0;
}

static int getpts(int v,int c)
{
    int i;

    for( i=0; i<numcst; i++ )
        if( cellstarttimes[i].vob==v && cellstarttimes[i].cell==c )
            return cellstarttimes[i].pts;
    return -1;
}

static void printtime(int v,int c)
{
    int t1=getpts(v,c),t2=getpts(v,1);
    if( t1>=0 && t2>=0 ) {
        int t=t1-t2;
        fprintf(xml,"%d:%02d:%02d.%03d",
                (t/90/1000/60/60),
                (t/90/1000/60)%60,
                (t/90/1000)%60,
                (t/90)%1000);
        return;
    }
    fprintf(xml,"-1");
}

static char *chap_xml_type[3]={""," chapter=\"1\""," program=\"1\""};

static int getprogramtype(vts_ptt_srpt_t *tt,pgc_t *p,int pn,int c)
{
    int ptype=0, i, j, pg=0;

    for( i=0; i<p->nr_of_programs; i++ )
        if( c==p->program_map[i] ) {
            pg=i+1;
            ptype=2;
            break;
        }
    if( ptype && tt ) {
        for( i=0; i<tt->nr_of_srpts; i++ )
            for( j=0; j<tt->title[i].nr_of_ptts; j++ )
                if( tt->title[i].ptt[j].pgcn==pn &&
                    tt->title[i].ptt[j].pgn==pg )
                    return 1;
    }
    return ptype;
}

static void dump_pgcs(ifo_handle_t *ifo,pgcit_t *pgcs,int titleset,int titlef)
{
    if( pgcs ) {
        int i, j;

        for( i=0; i<pgcs->nr_of_pgci_srp; i++ ) {
            pgc_t *p=pgcs->pgci_srp[i].pgc;

            fprintf(xml,"\t    <pgc>\n");
            for( j=0; j<p->nr_of_cells; j++ ) {
                fprintf(xml,"\t\t<vob file=\"vob_%02d_%03d%c.vob\">\n\t\t    <cell start=\"",titleset,p->cell_position[j].vob_id_nr,titlef?'t':'m');
                printtime(p->cell_position[j].vob_id_nr,p->cell_position[j].cell_nr);
                fprintf(xml,"\" end=\"");
                printtime(p->cell_position[j].vob_id_nr,p->cell_position[j].cell_nr+1);
                fprintf(xml,"\"%s />\n\t\t</vob>\n",chap_xml_type[getprogramtype(titlef?ifo->vts_ptt_srpt:0,p,i+1,j+1)]);
            }
            fprintf(xml,"\t    </pgc>\n");
        }
    }
}

static void dvddump(dvd_reader_t *dvd,int titleset,int titlef)
{
    ifo_handle_t *ifo;
    dvd_file_t *vobs;
    c_adt_t *cptr;
    cell_adr_t *cells;
    int numcells,i,j;

    if( titleset<0 || titleset>9 ) {
        fprintf(stderr,"ERR:  Can only handle titlesets 0..9\n");
        exit(1);
    }
    if( titlef<0 || titlef>1 ) {
        fprintf(stderr,"ERR:  Title flag must be 0 (menu) or 1 (title)\n");
        exit(1);
    }
    if (titlef==1 && titleset==0) {
        fprintf(stderr,"ERR:  No title for VMGM\n");
        exit(1);
    }
    ifo=ifoOpen(dvd,titleset);
    if( !titleset )
        numtitlesets=ifo->vmgi_mat->vmg_nr_of_title_sets;
    cptr=titlef?ifo->vts_c_adt:ifo->menu_c_adt;
    if( cptr ) {
        cells=cptr->cell_adr_table;
        numcells=(cptr->last_byte+1-C_ADT_SIZE)/sizeof(cell_adr_t);
    } else {
        cells=0;
        numcells=0;
    }
    numcst=0;

    vobs=DVDOpenFile(dvd,titleset,titlef?DVD_READ_TITLE_VOBS:DVD_READ_MENU_VOBS);
    
    for( i=0; i<numcells; i++ ) {
        char fname[100];
        int h,b;

        sprintf(fname,"vob_%02d_%03d%c.vob",titleset,cells[i].vob_id,titlef?'t':'m');
        if( vobexists(cells,i,cells[i].vob_id) )
            h=open(fname,O_CREAT|O_APPEND|O_WRONLY,0666);
        else
            h=open(fname,O_CREAT|O_TRUNC|O_WRONLY,0666);
        
        fprintf(stderr,"[%d] VOB %d, Cell %d, Size %d kbytes\n",i,cells[i].vob_id,cells[i].cell_id,(cells[i].last_sector-cells[i].start_sector+1)*DVD_VIDEO_LB_LEN/1024);
        if( h<0 ) {
            fprintf(stderr,"ERR:  Cannot open %s for writing\n",fname);
            exit(1);
        }
        for( b=cells[i].start_sector; b<=cells[i].last_sector; b+=BIGBLOCKSECT ) {
            int rl=cells[i].last_sector+1-b;
            if( rl > BIGBLOCKSECT ) rl = BIGBLOCKSECT;
            if( DVDReadBlocks(vobs,b,rl,bigblock) < rl ) {
                fprintf(stderr,"ERR:  Error reading data: %s\n",strerror(errno));
                exit(1);
            }
            for( j=0; j<rl; j++ ) {
                if( bigblock[j*DVD_VIDEO_LB_LEN+14] == 0 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+15] == 0 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+16] == 1 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+17] == 0xbb && // system header
                    bigblock[j*DVD_VIDEO_LB_LEN+38] == 0 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+39] == 0 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+40] == 1 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+41] == 0xbf && // 1st private2
                    bigblock[j*DVD_VIDEO_LB_LEN+1024] == 0 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+1025] == 0 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+1026] == 1 &&
                    bigblock[j*DVD_VIDEO_LB_LEN+1027] == 0xbf ) // 2nd private2
                {
                    unsigned char *pts=bigblock+j*DVD_VIDEO_LB_LEN+0x39;
                    addcst(cells[i].vob_id,cells[i].cell_id,(pts[0]<<24)|(pts[1]<<16)|(pts[2]<<8)|pts[3]);
                    break;
                }
            }
            if( write(h,bigblock,rl*DVD_VIDEO_LB_LEN) < rl*DVD_VIDEO_LB_LEN ) {
                fprintf(stderr,"ERR:  Error writing data: %s\n",strerror(errno));
                exit(1);
            }
        }
        close(h);
    }
    if( titlef ) {
        fprintf(xml,"\t<titles>\n");
        dump_pgcs(ifo,ifo->vts_pgcit,titleset,titlef);
        fprintf(xml,"\t</titles>\n");
    } else {
        if( ifo->pgci_ut ) {
            for( i=0; i<ifo->pgci_ut->nr_of_lus; i++ ) {
                pgci_lu_t *lu=&ifo->pgci_ut->lu[i];
                if( lu->lang_code )
                    fprintf(xml,"\t<menus lang=\"%c%c\">\n",lu->lang_code>>8,lu->lang_code);
                else
                    fprintf(xml,"\t<menus>\n");
                dump_pgcs(ifo,lu->pgcit,titleset,titlef);
                fprintf(xml,"\t</menus>\n");
            }
        }
    }

    ifoClose(ifo);
    fflush(xml);
}

int main(int argc,char **argv)
{
    dvd_reader_t *dvd;
    int i;

    if( argc!=2) {
        fprintf(stderr,"syntax: dvdunauthor path\n");
        return 0;
    }
    dvd=DVDOpen(argv[1]);
    if(!dvd) {
        fprintf(stderr,"ERR:  Cannot open path '%s'\n",argv[1]);
        return 1;
    }
    xml=fopen("dvdauthor.xml","wt");
    fprintf(xml,"<dvdauthor>\n");
    for( i=0; i<=numtitlesets; i++ ) {
        if( i )
            fprintf(stderr,"MENU %d\n",i);
        else
            fprintf(stderr,"VMGM\n");
        fprintf(xml,i?"    <titleset>\n":"    <vmgm>\n");
        dvddump(dvd,i,0);
        if( i ) {
            fprintf(stderr,"TITLE %d\n",i);
            dvddump(dvd,i,1);
        }
        fprintf(xml,i?"    </titleset>\n":"    </vmgm>\n");
    }
    fprintf(xml,"</dvdauthor>\n");
    fclose(xml);
    DVDClose(dvd);
    return 0;
}
